const mongoose = require("mongoose");
const crypto = require("crypto");

const certificateSchema = new mongoose.Schema(
  {
    student: {
      type: mongoose.Schema.ObjectId,
      ref: "User",
      required: [true, "Certificate must belong to a student"],
    },
    course: {
      type: mongoose.Schema.ObjectId,
      ref: "Course",
      required: [true, "Certificate must belong to a course"],
    },
    // Snapshot of data at time of issuance (in case User/Course changes later)
    studentName: String,
    courseName: String,
    instructors: [String], // Array of instructor names

    issueDate: {
      type: Date,
      default: Date.now,
    },
    certificateId: {
      type: String,
      unique: true,
    },
  },
  { timestamps: true }
);

// 1. Generate Unique ID before saving
certificateSchema.pre("save", function (next) {
  if (!this.certificateId) {
    // Generates a code like: CERT-9A3F-1B2C
    this.certificateId = `CERT-${crypto
      .randomBytes(3)
      .toString("hex")
      .toUpperCase()}`;
  }
  next();
});

// 2. Prevent duplicate certificates
certificateSchema.index({ student: 1, course: 1 }, { unique: true });

const Certificate = mongoose.model("Certificate", certificateSchema);
module.exports = Certificate;
